import React, { useState, useEffect } from "react";
import {
  FaExclamationTriangle,
  FaFileAlt,
  FaSpinner,
  FaChevronLeft,
  FaChevronRight
} from "react-icons/fa";
import { useTheme } from "../../contexts/ThemeContext";
import { buildApiUrl, API_ENDPOINTS } from "../../utils/config";

const MissingFormsTable = () => {
  const [missingForms, setMissingForms] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage] = useState(5);
  const { isLight } = useTheme();

  // Theme-based styling
  const cardBgClass = isLight
    ? "bg-white border border-yellow-200 shadow-md hover:shadow-lg"
    : "bg-slate-800 border border-slate-700 shadow-xl hover:shadow-2xl";

  const headingClass = isLight ? "text-amber-800" : "text-slate-100";
  const textClass = isLight ? "text-amber-700" : "text-slate-300";

  // Fetch missing forms data
  const fetchMissingForms = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch(
        buildApiUrl(API_ENDPOINTS.PROJECT_MISSING_FORMS)
      );

      if (!response.ok) {
        throw new Error(`Failed to fetch missing forms: ${response.status}`);
      }

      const data = await response.json();
      setMissingForms(data);
    } catch (err) {
      console.error("Error fetching missing forms:", err);
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchMissingForms();
  }, []);

  // Reset to first page when data changes
  useEffect(() => {
    setCurrentPage(1);
  }, [missingForms.length]);

  // Pagination logic
  const indexOfLastItem = currentPage * itemsPerPage;
  const indexOfFirstItem = indexOfLastItem - itemsPerPage;
  const currentMissingForms = missingForms.slice(
    indexOfFirstItem,
    indexOfLastItem
  );
  const totalPages = Math.ceil(missingForms.length / itemsPerPage);

  const goToPage = (page) => {
    setCurrentPage(page);
  };

  const goToPrevious = () => {
    setCurrentPage((prev) => Math.max(prev - 1, 1));
  };

  const goToNext = () => {
    setCurrentPage((prev) => Math.min(prev + 1, totalPages));
  };

  if (loading) {
    return (
      <div
        className={`${cardBgClass} rounded-xl overflow-hidden transition-all duration-300`}
      >
        <div className="p-4 sm:p-8 text-center">
          <div className={`${textClass}`}>
            <div className="animate-spin h-8 w-8 border-4 border-blue-600 border-t-transparent rounded-full mx-auto mb-4"></div>
            <p className="text-base sm:text-lg font-medium">Loading missing forms...</p>
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div
        className={`${cardBgClass} rounded-xl overflow-hidden transition-all duration-300`}
      >
        <div className="p-4 sm:p-8 text-center">
          <div className={`${textClass}`}>
            <div className="text-red-500 text-4xl mb-4">⚠️</div>
            <p className="text-base sm:text-lg font-medium mb-2 text-red-500">
              Error Loading Data
            </p>
            <p className="text-sm opacity-75">{error}</p>
            <button
              onClick={fetchMissingForms}
              className="mt-4 px-4 py-2 bg-gradient-to-r from-yellow-500 to-amber-500 hover:from-yellow-600 hover:to-amber-600 text-white rounded-lg transition-all duration-200 text-sm"
            >
              Try Again
            </button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div
      className={`${cardBgClass} rounded-xl overflow-hidden transition-all duration-300`}
    >
      {/* Header */}
      <div className={`px-4 sm:px-6 py-4 border-b ${
        isLight ? "border-gray-200" : "border-slate-600"
      }`}>
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-3 sm:space-y-0">
          <div className="flex items-center space-x-3">
            <div
              className={`p-2 sm:p-3 rounded-lg flex items-center justify-center ${
                isLight ? "bg-yellow-100" : "bg-gray-700"
              }`}
            >
              <FaFileAlt
                className={`h-5 w-5 sm:h-6 sm:w-6 ${
                  isLight ? "text-amber-600" : "text-yellow-400"
                } flex-shrink-0`}
              />
            </div>
            <div>
              <h2 className={`text-lg sm:text-xl font-bold ${headingClass}`}>
                Missing Documentation Form
              </h2>
              <p
                className={`text-xs sm:text-sm ${
                  isLight ? "text-gray-600" : "text-gray-400"
                }`}
              >
                Projects that require documentation form submission
              </p>
            </div>
          </div>
          
          {/* Count badge */}
          <div className="flex items-center justify-center sm:justify-end">
            <div
              className={`px-3 py-1 rounded-full text-sm font-medium ${
                isLight ? "bg-red-100 text-red-800" : "bg-red-900 text-red-200"
              }`}
            >
              {missingForms.length} Missing
            </div>
          </div>
        </div>
      </div>

      {/* Content */}
      {missingForms.length === 0 ? (
        <div className="p-4 sm:p-8 text-center">
          <div className={`${textClass}`}>
            <div className="text-4xl mb-4">📋</div>
            <p className="text-base sm:text-lg font-medium mb-2">All Forms Submitted!</p>
            <p className="text-sm opacity-75">
              Great job! All projects have their kickoff forms submitted.
            </p>
          </div>
        </div>
      ) : (
        <div className="overflow-x-auto">
          <table className="w-full table-fixed">
            <thead
              className={`${
                isLight
                  ? "bg-gradient-to-r from-yellow-200 to-amber-200"
                  : "bg-slate-700"
              }`}
            >
              <tr>
                <th
                  className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                  style={{ width: "40%" }}
                >
                  Project Name
                </th>
                <th
                  className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                  style={{ width: "30%" }}
                >
                  Status
                </th>
                <th
                  className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                  style={{ width: "30%" }}
                >
                  Project Lead
                </th>
              </tr>
            </thead>
            <tbody className={isLight ? "bg-white" : "bg-gray-800"}>
              {currentMissingForms.map((project, index) => (
                <tr
                  key={index}
                  className={`border-b transition-colors duration-200 ${
                    isLight
                      ? "border-gray-200 hover:bg-yellow-50"
                      : "border-slate-600 hover:bg-slate-700"
                  } ${
                    index % 2 === 0
                      ? isLight
                        ? "bg-gray-50"
                        : "bg-gray-750"
                      : isLight
                      ? "bg-white"
                      : "bg-gray-800"
                  }`}
                >
                  <td
                    className={`px-2 py-2 text-center font-medium text-xs truncate ${textClass}`}
                    title={project.project_name}
                  >
                    {project.project_name}
                  </td>
                  <td className={`px-2 py-2 text-center text-xs`}>
                    <span
                      className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                        isLight
                          ? "bg-red-100 text-red-800"
                          : "bg-red-900 text-red-200"
                      }`}
                    >
                      Missing Form
                    </span>
                  </td>
                  <td
                    className={`px-2 py-2 text-center text-xs truncate ${textClass}`}
                    title={project.project_lead_name || "Not Assigned"}
                  >
                    {project.project_lead_name || "Not Assigned"}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

      {/* Pagination Controls */}
      {totalPages > 1 && (
        <div
          className={`mt-0 px-3 py-3 border-t overflow-hidden ${
            isLight
              ? "bg-gray-50 border-gray-200"
              : "bg-slate-700 border-slate-600"
          }`}
        >
          <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-3 sm:gap-2 min-w-0">
            {/* Page Info */}
            <div
              className={`text-xs flex-shrink-0 text-center sm:text-left ${
                isLight ? "text-gray-600" : "text-gray-300"
              }`}
            >
              {indexOfFirstItem + 1}-
              {Math.min(indexOfLastItem, missingForms.length)} of{" "}
              {missingForms.length}
            </div>

            {/* Pagination Controls */}
            <div className="flex items-center justify-center gap-2">
              {/* Previous Button */}
              <button
                onClick={goToPrevious}
                disabled={currentPage === 1}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === 1
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <FaChevronLeft className="text-xs flex-shrink-0" />
                <span>Prev</span>
              </button>

              {/* Current Page */}
              <div className={`px-3 py-2 rounded-lg font-medium text-sm ${
                isLight ? 'bg-amber-500 text-white' : 'bg-gray-300 text-gray-900'
              }`}>
                {currentPage}
              </div>

              {/* Next Button */}
              <button
                onClick={goToNext}
                disabled={currentPage === totalPages}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === totalPages
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <span>Next</span>
                <FaChevronRight className="text-xs flex-shrink-0" />
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default MissingFormsTable;
