// Form persistence utility functions
// This file provides helper functions for persisting form data in localStorage

/**
 * Get persisted form data from localStorage
 * @param {string} formType - The type of form (e.g., 'addProject', 'editEmployee')
 * @param {string|number} entityId - Optional entity ID for edit forms
 * @returns {object|null} - The persisted form data or null if not found
 */
export const getPersistedFormData = (formType, entityId = null) => {
  try {
    const key = entityId ? `formData_${formType}_${entityId}` : `formData_${formType}`;
    const persisted = localStorage.getItem(key);
    return persisted ? JSON.parse(persisted) : null;
  } catch (error) {
    console.error("Error getting persisted form data:", error);
    return null;
  }
};

/**
 * Save form data to localStorage
 * @param {string} formType - The type of form (e.g., 'addProject', 'editEmployee')
 * @param {object} data - The form data to persist
 * @param {string|number} entityId - Optional entity ID for edit forms
 */
export const saveFormData = (formType, data, entityId = null) => {
  try {
    const key = entityId ? `formData_${formType}_${entityId}` : `formData_${formType}`;
    localStorage.setItem(key, JSON.stringify(data));
  } catch (error) {
    console.error("Error saving form data:", error);
  }
};

/**
 * Clear persisted form data from localStorage
 * @param {string} formType - The type of form (e.g., 'addProject', 'editEmployee')
 * @param {string|number} entityId - Optional entity ID for edit forms
 */
export const clearPersistedFormData = (formType, entityId = null) => {
  try {
    const key = entityId ? `formData_${formType}_${entityId}` : `formData_${formType}`;
    localStorage.removeItem(key);
  } catch (error) {
    console.error("Error clearing persisted form data:", error);
  }
};

/**
 * Check if form data should be persisted (has meaningful content)
 * @param {object} formData - The form data to check
 * @param {string} excludeField - Optional field to exclude from check (e.g., 'user_id')
 * @returns {boolean} - True if form data should be persisted
 */
export const shouldPersistFormData = (formData, excludeField = null) => {
  return Object.entries(formData).some(([key, value]) => {
    if (excludeField && key === excludeField) return false;
    // For dropdown fields, even "0" values should be considered meaningful
    if (key.includes('_id') || key.includes('emp_id')) {
      return value !== "" && value !== null && value !== undefined;
    }
    return value !== "" && value !== null && value !== undefined;
  });
};

/**
 * Hook-like function to manage form persistence
 * @param {string} formType - The type of form
 * @param {object} formData - Current form data
 * @param {boolean} isOpen - Whether the form is open
 * @param {string|number} entityId - Optional entity ID for edit forms
 * @returns {object} - Object with functions to manage persistence
 */
export const useFormPersistence = (formType, formData, isOpen, entityId = null) => {
  const loadPersistedData = () => {
    if (isOpen) {
      return getPersistedFormData(formType, entityId);
    }
    return null;
  };

  const saveCurrentData = () => {
    if (isOpen && shouldPersistFormData(formData)) {
      saveFormData(formType, formData, entityId);
    }
  };

  const clearData = () => {
    clearPersistedFormData(formType, entityId);
  };

  return {
    loadPersistedData,
    saveCurrentData,
    clearData
  };
};
