import React from 'react';
import { useTheme } from '../../contexts/ThemeContext';
import { FaEye, FaDownload } from 'react-icons/fa';

const ProjectCard = ({ project, onView, onDownload }) => {
  const { isLight } = useTheme();

  const cardBgClass = isLight 
    ? 'bg-white border border-yellow-200 shadow-md hover:shadow-lg active:shadow-md' 
    : 'bg-gradient-to-br from-gray-800 to-slate-800 border border-gray-700 shadow-lg hover:shadow-xl active:shadow-lg';
  
  const headingClass = isLight 
    ? 'text-amber-800' 
    : 'text-gray-100';
  
  const textClass = isLight 
    ? 'text-amber-700' 
    : 'text-gray-300';

  const handleView = () => {
    if (onView) onView(project);
  };

  const handleDownload = () => {
    if (onDownload) onDownload(project);
  };



  return (
    <div
      className={`${cardBgClass} rounded-xl p-4 transition-all duration-200 ${
        isLight ? 'shadow-md hover:shadow-lg' : 'shadow-lg hover:shadow-xl'
      }`}
    >
      {/* Header Section with Project Name */}
      <div className="mb-3">
        <h3 className={`font-bold text-base leading-tight mb-1 ${
          isLight ? 'text-amber-800' : 'text-gray-100'
        }`} title={project.project_name}>
          {project.project_name}
        </h3>
        <div className="flex justify-between items-center">
          <span className={`text-xs font-mono ${
            isLight ? 'text-amber-600' : 'text-gray-300'
          }`}>
            ID: {project.project_id}
          </span>
          <span className={`text-xs ${
            isLight ? 'text-gray-600' : 'text-gray-400'
          }`}>
            Lead: {project.project_lead_name || 'Not Assigned'}
          </span>
        </div>
      </div>

      {/* Project Information - Compact Layout */}
      <div className="space-y-2 mb-3">
        {/* Type */}
        <div className="flex justify-between items-center">
          <span className={`text-xs font-medium ${textClass}`}>Type</span>
          <span className={`text-xs px-2 py-1 rounded font-medium ${
            isLight 
              ? 'bg-blue-100 text-blue-800' 
              : 'bg-blue-800 text-blue-200'
          }`}>
            {project.type_name || 'Not Assigned'}
          </span>
        </div>

        {/* Status */}
        <div className="flex justify-between items-center">
          <span className={`text-xs font-medium ${textClass}`}>Status</span>
          <span className={`text-xs px-2 py-1 rounded font-medium ${
            isLight 
              ? 'bg-green-100 text-green-800' 
              : 'bg-green-800 text-green-200'
          }`}>
            {project.status_name || 'Not Assigned'}
          </span>
        </div>

        {/* Progress */}
        <div className="flex justify-between items-center">
          <span className={`text-sm font-medium ${textClass}`}>PO Status:</span>
          <span
            className={`text-sm px-3 py-1 rounded whitespace-nowrap ${
              project.po_status === "PO Received"
                ? isLight
                  ? "bg-green-100 text-green-800"
                  : "bg-green-800 text-green-200"
                : project.po_status === "PO Not Received"
                ? isLight
                  ? "bg-yellow-100 text-yellow-800"
                  : "bg-yellow-800 text-yellow-200"
                : isLight
                ? "bg-gray-100 text-gray-800"
                : "bg-gray-700 text-gray-200"
            }`}
          >
            {project.po_status || "Not Set"}
          </span>
        </div>

        {/* Hours */}
        <div className="flex justify-between items-center">
          <span className={`text-xs font-medium ${textClass}`}>Hours</span>
          <div className="flex items-center gap-2">
            <span className={`text-xs ${isLight ? 'text-gray-600' : 'text-gray-400'}`}>
              Est: {project.estimated_hours || 0}h
            </span>
            <span className={`text-xs font-medium ${isLight ? 'text-gray-800' : 'text-gray-200'}`}>
              Act: {project.actual_hours || 0}h
            </span>
          </div>
        </div>

        {/* Deadline */}
        <div className="flex justify-between items-center">
          <span className={`text-xs font-medium ${textClass}`}>Deadline</span>
          <span className={`text-xs px-2 py-1 rounded font-medium ${
            isLight 
              ? 'bg-gray-100 text-gray-800' 
              : 'bg-gray-700 text-gray-200'
          }`}>
            {project.deadline || 'Not Set'}
          </span>
        </div>

        {/* Remarks */}
        {project.remarks && (
          <div className="flex justify-between items-start gap-3">
            <span className={`text-xs font-medium ${textClass} flex-shrink-0`}>Remarks</span>
            <div className={`text-xs max-w-[65%] ${
              isLight ? 'text-gray-600' : 'text-gray-300'
            }`}>
              <p className="truncate" title={project.remarks}>
                {project.remarks}
              </p>
            </div>
          </div>
        )}
      </div>

      {/* View Details Button - Below all sections */}
      <button 
        className={`w-full px-3 py-2 rounded-lg transition-all duration-200 active:scale-95 sm:hover:scale-105 flex items-center justify-center gap-2 text-xs font-medium touch-manipulation ${
          isLight 
            ? 'bg-amber-500 hover:bg-amber-600 active:bg-amber-700 text-white shadow-sm' 
            : 'bg-blue-600 hover:bg-blue-700 active:bg-blue-800 text-white shadow-md'
        }`}
        onClick={handleView}
        title="View Details"
      >
        <FaEye size={12} className="flex-shrink-0" />
        <span>View Details</span>
      </button>

      {/* Floating Download Button - Sticky Bottom Right (Mobile Only) */}
      {onDownload && (
        <button
          onClick={handleDownload}
          className={`fixed bottom-6 right-6 w-14 h-14 rounded-full shadow-lg transition-all duration-200 active:scale-95 hover:scale-110 flex items-center justify-center z-50 md:hidden ${
            isLight
              ? 'bg-green-500 hover:bg-green-600 active:bg-green-700 text-white shadow-green-200'
              : 'bg-green-600 hover:bg-green-700 active:bg-green-800 text-white shadow-green-900'
          }`}
          title="Download Project Data"
          style={{
            boxShadow: isLight 
              ? '0 8px 25px -5px rgba(34, 197, 94, 0.3), 0 8px 10px -6px rgba(34, 197, 94, 0.2)' 
              : '0 8px 25px -5px rgba(34, 197, 94, 0.4), 0 8px 10px -6px rgba(34, 197, 94, 0.3)'
          }}
        >
          <FaDownload size={16} className="flex-shrink-0" />
        </button>
      )}
    </div>
  );
};

export default ProjectCard; 