import { useEffect, useRef } from 'react';
import './UnityGame.css';

const UnityGame = () => {
  const canvasRef = useRef(null);
  const unityInstanceRef = useRef(null);

  useEffect(() => {
    const canvas = canvasRef.current;
    if (!canvas) return;

    // Banner function for Unity messages
    const unityShowBanner = (msg, type) => {
      const warningBanner = document.querySelector("#unity-warning");
      if (!warningBanner) return;

      const updateBannerVisibility = () => {
        warningBanner.style.display = warningBanner.children.length ? 'block' : 'none';
      };

      const div = document.createElement('div');
      div.innerHTML = msg;
      warningBanner.appendChild(div);

      if (type === 'error') {
        div.style = 'background: red; padding: 10px;';
      } else {
        if (type === 'warning') div.style = 'background: yellow; padding: 10px;';
        setTimeout(() => {
          warningBanner.removeChild(div);
          updateBannerVisibility();
        }, 5000);
      }
      updateBannerVisibility();
    };

    const buildUrl = "/v2/Build";
    const loaderUrl = buildUrl + "/v2.loader.js";
    const config = {
      arguments: [],
      dataUrl: buildUrl + "/v2.data",
      frameworkUrl: buildUrl + "/v2.framework.js",
      codeUrl: buildUrl + "/v2.wasm",
      streamingAssetsUrl: "StreamingAssets",
      companyName: "eframe",
      productName: "2D Platformer",
      productVersion: "1.0",
      showBanner: unityShowBanner,
    };

    // Detect mobile devices
    if (/iPhone|iPad|iPod|Android/i.test(navigator.userAgent)) {
      const container = document.querySelector("#unity-container");
      if (container) {
        container.className = "unity-mobile";
        canvas.className = "unity-mobile";
      }
    } else {
      canvas.style.width = "960px";
      canvas.style.height = "600px";
    }

    // Show loading bar
    const loadingBar = document.querySelector("#unity-loading-bar");
    if (loadingBar) loadingBar.style.display = "block";

    // Load Unity
    const script = document.createElement("script");
    script.src = loaderUrl;
    script.onload = () => {
      window.createUnityInstance(canvas, config, (progress) => {
        const progressBar = document.querySelector("#unity-progress-bar-full");
        if (progressBar) {
          progressBar.style.width = 100 * progress + "%";
        }
      }).then((unityInstance) => {
        unityInstanceRef.current = unityInstance;
        const loadingBar = document.querySelector("#unity-loading-bar");
        if (loadingBar) loadingBar.style.display = "none";

        const fullscreenButton = document.querySelector("#unity-fullscreen-button");
        if (fullscreenButton) {
          fullscreenButton.onclick = () => {
            unityInstance.SetFullscreen(1);
          };
        }
      }).catch((message) => {
        console.error("Unity loading error:", message);
        alert(message);
      });
    };

    script.onerror = () => {
      console.error("Failed to load Unity loader script");
      alert("Failed to load Unity game. Please check if all files are in place.");
    };

    document.body.appendChild(script);

    // Cleanup
    return () => {
      if (unityInstanceRef.current) {
        try {
          unityInstanceRef.current.Quit();
        } catch (e) {
          console.log("Unity instance cleanup:", e);
        }
      }
      if (script && script.parentNode) {
        script.parentNode.removeChild(script);
      }
    };
  }, []);

  return (
    <div id="unity-container" className="unity-desktop">
      <canvas 
        id="unity-canvas" 
        ref={canvasRef}
        width="960" 
        height="600" 
        tabIndex="-1"
      />
      <div id="unity-loading-bar">
        <div id="unity-logo"></div>
        <div id="unity-progress-bar-empty">
          <div id="unity-progress-bar-full"></div>
        </div>
      </div>
      <div id="unity-warning"></div>
      <div id="unity-footer">
        <div id="unity-logo-title-footer"></div>
        <div id="unity-fullscreen-button"></div>
        <div id="unity-build-title">2D Platformer</div>
      </div>
    </div>
  );
};

export default UnityGame;

